<?php
/**
 * Plugin Name: Trapz Hot Bank
 * Description: A banking system with multiple account types and features.
 * Version: 1.0
 * Author: Jones Lawson Trap-Z
 */

define('TRAPZ_HOT_BANK_VERSION', '1.0');

register_activation_hook(__FILE__, 'trapz_hot_bank_activate');
function trapz_hot_bank_activate() {
    global $wpdb;
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    $charset_collate = $wpdb->get_charset_collate();

    // User Accounts Table
    $accounts_table = $wpdb->prefix . 'trapz_hot_bank_accounts';
    $sql_accounts = "CREATE TABLE $accounts_table (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        account_type varchar(50) NOT NULL,
        account_number varchar(50) NOT NULL,
        balance decimal(10,2) DEFAULT 0.00 NOT NULL,
        imf_code varchar(50),
        PRIMARY KEY  (id),
        UNIQUE KEY account_number (account_number)
    ) $charset_collate;";

    $sql_pending_transfers = "CREATE TABLE {$wpdb->prefix}trapz_hot_bank_pending_transfers (
        id INT(11) NOT NULL AUTO_INCREMENT,
        user_id BIGINT(20) NOT NULL,
        recipient_name VARCHAR(255) NOT NULL,
        recipient_account_number VARCHAR(50) NOT NULL,
        account_type VARCHAR(50) NOT NULL,
        amount DECIMAL(10,2) NOT NULL,
        currency VARCHAR(10) NOT NULL,
        status VARCHAR(20) NOT NULL DEFAULT 'pending',
        PRIMARY KEY (id)
    ) $charset_collate;";

    // Transaction History Table
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';
    $sql_transactions = "CREATE TABLE $transactions_table (
        id int(11) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        reference varchar(255) NOT NULL,
        date datetime NOT NULL,
        type enum('debit','credit') NOT NULL,
        amount decimal(10,2) NOT NULL,
        currency varchar(10) NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    // Bitcoin Transactions Table
    $btc_transactions_table = $wpdb->prefix . 'trapz_hot_bank_btc_transactions';
    $sql_btc_transactions = "CREATE TABLE $btc_transactions_table (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        btc_amount decimal(10,8) NOT NULL,
        purchase_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        profit_rate decimal(3,2) NOT NULL,
        expiry_date datetime NOT NULL,
        status varchar(50) NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";

    // Execute the SQL commands
    dbDelta($sql_accounts);
    dbDelta($sql_transactions);
    dbDelta($sql_btc_transactions);
    dbDelta($sql_pending_transfers);

    add_option('trapz_hot_bank_version', TRAPZ_HOT_BANK_VERSION);
}




// User Profile Fields: Account Details and Currency
add_action('show_user_profile', 'trapz_hot_bank_user_profile_fields');
add_action('edit_user_profile', 'trapz_hot_bank_user_profile_fields');
function trapz_hot_bank_user_profile_fields($user) {
    $account_types = ['savings', 'checking', 'fixed_deposit'];
    $current_currency = get_the_author_meta('currency', $user->ID);
    $currencies = ["$", "€", "¥", "£", "A$", "C$", "CHF", "¥", "kr", "NZ$", "₦", "₹", "د.إ", "د.ك", "GH₵", "R"];

    echo '<h3>' . __("Bank Account Information", "trapz-hot-bank") . '</h3>';
    echo '<table class="form-table">';

    foreach ($account_types as $type) {
        trapz_hot_bank_create_account_field($user, $type);
    }

    echo '<tr><th><label for="currency">' . __("Preferred Currency") . '</label></th>';
    echo '<td><select name="currency" id="currency">';
    foreach ($currencies as $currency) {
        echo '<option value="' . $currency . '"' . selected($current_currency, $currency, false) . '>' . $currency . '</option>';
    }
    echo '</select><span class="description">' . __("Select your preferred currency.") . '</span></td></tr>';
    echo '</table>';
}

function trapz_hot_bank_create_account_field($user, $type) {
    echo '<tr><th><label for="' . $type . '-account">' . __(ucfirst($type) . " Account Balance") . '</label></th>';
    echo '<td><input type="text" name="' . $type . '_account" id="' . $type . '-account" value="' . esc_attr(get_the_author_meta($type . '_account', $user->ID)) . '" class="regular-text" /><br />';
    echo '<span class="description">' . __("Please enter your " . $type . " account balance.") . '</span></td></tr>';
    echo '<tr><th><label for="' . $type . '-account-number">' . __(ucfirst($type) . " Account Number") . '</label></th>';
    echo '<td><input type="text" name="' . $type . '_account_number" id="' . $type . '-account-number" value="' . esc_attr(get_the_author_meta($type . '_account_number', $user->ID)) . '" class="regular-text" /><br />';
    echo '<span class="description">' . __("Your " . $type . " account number.") . '</span></td></tr>';
}

// Save User Profile Fields
add_action('personal_options_update', 'trapz_hot_bank_save_user_profile_fields');
add_action('edit_user_profile_update', 'trapz_hot_bank_save_user_profile_fields');
function trapz_hot_bank_save_user_profile_fields($user_id) {
    if (!current_user_can('edit_user', $user_id)) {
        return false;
    }

    $account_types = ['savings', 'checking', 'fixed_deposit'];
    foreach ($account_types as $type) {
        update_user_meta($user_id, $type . '_account', $_POST[$type . '_account']);
        update_user_meta($user_id, $type . '_account_number', $_POST[$type . '_account_number']);
    }

    if (isset($_POST['currency'])) {
        update_user_meta($user_id, 'currency', $_POST['currency']);
    }
}

// Shortcodes for Displaying Account Info and Currency
function trapz_hot_bank_get_user_account_balance($account_type) {
    $user_id = get_current_user_id();
    if (!$user_id) return '';

    $balance = get_user_meta($user_id, "{$account_type}_account", true);
    return number_format((float)$balance, 2, '.', ',');
}

add_shortcode('savings_balance', 'trapz_hot_bank_savings_balance');
function trapz_hot_bank_savings_balance() {
    return trapz_hot_bank_get_user_account_balance('savings');
}

add_shortcode('checking_balance', 'trapz_hot_bank_checking_balance');
function trapz_hot_bank_checking_balance() {
    return trapz_hot_bank_get_user_account_balance('checking');
}

add_shortcode('fixed_deposit_balance', 'trapz_hot_bank_fixed_deposit_balance');
function trapz_hot_bank_fixed_deposit_balance() {
    return trapz_hot_bank_get_user_account_balance('fixed_deposit');
}

add_shortcode('total_account_balance', 'trapz_hot_bank_total_account_balance');
function trapz_hot_bank_total_account_balance() {
    $user_id = get_current_user_id();
    if (!$user_id) return '';

    $total_balance = 0;
    $account_types = ['savings', 'checking', 'fixed_deposit'];
    foreach ($account_types as $type) {
        $total_balance += (float) get_user_meta($user_id, "{$type}_account", true);
    }

    return number_format($total_balance, 2, '.', ',');
}

add_shortcode('user_currency', 'trapz_hot_bank_user_currency');
function trapz_hot_bank_user_currency() {
    $user_id = get_current_user_id();
    if (!$user_id) return '';
    return get_user_meta($user_id, 'currency', true);
}

// Shortcode for Last Three Digits of Savings Account Number
add_shortcode('savings_account_last3', 'trapz_hot_bank_savings_account_last3');
function trapz_hot_bank_savings_account_last3() {
    return trapz_hot_bank_get_user_account_last3('savings');
}

// Shortcode for Last Three Digits of Checking Account Number
add_shortcode('checking_account_last3', 'trapz_hot_bank_checking_account_last3');
function trapz_hot_bank_checking_account_last3() {
    return trapz_hot_bank_get_user_account_last3('checking');
}

// Shortcode for Last Three Digits of Fixed Deposit Account Number
add_shortcode('fixed_deposit_account_last3', 'trapz_hot_bank_fixed_deposit_account_last3');
function trapz_hot_bank_fixed_deposit_account_last3() {
    return trapz_hot_bank_get_user_account_last3('fixed_deposit');
}

function trapz_hot_bank_get_user_account_last3($account_type) {
    $user_id = get_current_user_id();
    if (!$user_id) return '';

    $account_number = get_user_meta($user_id, "{$account_type}_account_number", true);
    if (strlen($account_number) >= 3) {
        return substr($account_number, -3);
    } else {
        return $account_number; // If the number is shorter than 3 digits, return the whole number
    }
}

add_action('admin_menu', 'trapz_hot_bank_admin_menu');

function trapz_hot_bank_admin_menu() {
    add_menu_page('Manage User Accounts', 'Manage Accounts', 'manage_options', 'trapz-hot-bank-manage-accounts', 'trapz_hot_bank_manage_accounts_page', 'dashicons-money-alt', 6);
}

function trapz_hot_bank_manage_accounts_page() {
    $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'fund';
    ?>
    <h2 class="nav-tab-wrapper">
        <a href="?page=trapz-hot-bank-manage-accounts&tab=fund" class="nav-tab <?php echo $active_tab == 'fund' ? 'nav-tab-active' : ''; ?>">Fund Accounts</a>
        <a href="?page=trapz-hot-bank-manage-accounts&tab=debit" class="nav-tab <?php echo $active_tab == 'debit' ? 'nav-tab-active' : ''; ?>">Debit Accounts</a>
    </h2>
    <?php
    if ($active_tab == 'fund') {
        trapz_hot_bank_fund_accounts_form();
    } else {
        trapz_hot_bank_debit_accounts_form();
    }
}
function trapz_hot_bank_fund_accounts_form() {
    ?>
    <div class="wrap">
        <h1>Fund User Accounts</h1>
        <form method="post" action="">
            <?php wp_nonce_field('trapz_hot_bank_fund_accounts_action', 'trapz_hot_bank_fund_accounts_nonce'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Select User</th>
                    <td><?php wp_dropdown_users(['name' => 'user_id', 'who' => 'subscriber']); ?></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Account Type</th>
                    <td>
                        <select name="account_type">
                            <option value="savings">Savings</option>
                            <option value="checking">Checking</option>
                            <option value="fixed_deposit">Fixed Deposit</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Amount</th>
                    <td><input type="number" name="amount" step="0.01"/></td>
                </tr>
            </table>
            <input type="submit" class="button-primary" value="Fund Account"/>
        </form>
    </div>
    <?php
}

function trapz_hot_bank_debit_accounts_form() {
    ?>
    <div class="wrap">
        <h1>Debit User Accounts</h1>
        <form method="post" action="">
            <?php wp_nonce_field('trapz_hot_bank_debit_accounts_action', 'trapz_hot_bank_debit_accounts_nonce'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Select User</th>
                    <td><?php wp_dropdown_users(['name' => 'user_id', 'who' => 'subscriber']); ?></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Account Type</th>
                    <td>
                        <select name="account_type">
                            <option value="savings">Savings</option>
                            <option value="checking">Checking</option>
                            <option value="fixed_deposit">Fixed Deposit</option>
                        </select>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Amount</th>
                    <td><input type="number" name="amount" step="0.01"/></td>
                </tr>
            </table>
            <input type="submit" class="button-primary" value="Debit Account"/>
        </form>
    </div>
    <?php
}
add_action('admin_init', 'trapz_hot_bank_process_account_changes');

function trapz_hot_bank_process_account_changes() {
    if (isset($_POST['trapz_hot_bank_fund_accounts_nonce']) && wp_verify_nonce($_POST['trapz_hot_bank_fund_accounts_nonce'], 'trapz_hot_bank_fund_accounts_action')) {
        trapz_hot_bank_process_fund_account();
    } elseif (isset($_POST['trapz_hot_bank_debit_accounts_nonce']) && wp_verify_nonce($_POST['trapz_hot_bank_debit_accounts_nonce'], 'trapz_hot_bank_debit_accounts_action')) {
        trapz_hot_bank_process_debit_account();
    }
}

function trapz_hot_bank_process_fund_account() {
    // Check nonce and permissions
    // Assuming you have a security check here

    $user_id = intval($_POST['user_id']);
    $account_type = sanitize_text_field($_POST['account_type']);
    $amount = floatval($_POST['amount']);
    $currency = get_user_meta($user_id, 'currency', true);

    // Perform validation checks
    if ($amount <= 0 || $user_id <= 0) {
        // Handle invalid input
        return;
    }

    // Update the user's account balance
    $current_balance = get_user_meta($user_id, "{$account_type}_account", true);
    $new_balance = $current_balance + $amount;
    update_user_meta($user_id, "{$account_type}_account", $new_balance);

    // Record the transaction
    record_transaction($user_id, "Direct Deposit to {$account_type}", 'credit', $amount, $currency);

    // Send funding email
    trapz_hot_bank_send_funding_email($user_id, $amount, $new_balance, $account_type);

    // Optionally, redirect to a confirmation page or output a success message
}



function trapz_hot_bank_process_debit_account() {
    // Check nonce and permissions
    // Assuming you have a security check here

    $user_id = intval($_POST['user_id']);
    $account_type = sanitize_text_field($_POST['account_type']);
    $amount = floatval($_POST['amount']);
    $currency = get_user_meta($user_id, 'currency', true);

    // Perform validation checks
    if ($amount <= 0 || $user_id <= 0) {
        // Handle invalid input
        return;
    }

    // Update the user's account balance
    $current_balance = get_user_meta($user_id, "{$account_type}_account", true);
    $new_balance = max($current_balance - $amount, 0); // Prevent negative balance
    update_user_meta($user_id, "{$account_type}_account", $new_balance);

    // Record the transaction
    record_transaction($user_id, "Direct Reversal from {$account_type}", 'debit', $amount, $currency);

    // Send debiting email
    trapz_hot_bank_send_debiting_email($user_id, $amount, $new_balance, $account_type);

    // Optionally, redirect to a confirmation page or output a success message
}


function trapz_hot_bank_send_funding_email($user_id, $amount, $new_balance, $account_type) {
    $user_info = get_userdata($user_id);
    $user_email = $user_info->user_email;
    $user_display_name = $user_info->display_name;
    $currency_symbol = get_user_meta($user_id, 'currency', true);
    $formatted_amount = number_format((float)$amount, 2, '.', ',');
    $formatted_new_balance = number_format((float)$new_balance, 2, '.', ',');
    $website_name = get_bloginfo('name');
    $admin_email = get_bloginfo('admin_email');
    $account_type_name = ucfirst($account_type);

    $subject = "Direct Deposit Notification - " . $website_name;
    $headers = array('Content-Type: text/html; charset=UTF-8');

    $message = "<!DOCTYPE html>
    <html lang='en'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>Direct Deposit Notification</title>
    </head>
    <body>
        <p>Dear {$user_display_name},</p>
        <p>We are pleased to inform you that a direct deposit of <span style='color: green;'>{$currency_symbol}{$formatted_amount}</span> has been successfully credited to your {$account_type_name} account with {$website_name}. Below are the details of the transaction:</p>
        <ul>
            <li><strong>Transaction Date:</strong> " . date('Y-m-d H:i:s') . "</li>
            <li><strong>Amount Deposited:</strong> <span style='color: green;'>{$currency_symbol}{$formatted_amount}</span></li>
            <li><strong>{$account_type_name} Account Balance:</strong> <span style='color: green;'>{$currency_symbol}{$formatted_new_balance}</span></li>
        </ul>
        <p>Please verify this information by reviewing your account statement. For questions, contact our customer service team at {$admin_email}.</p>
        <p>Thank you for choosing {$website_name}.</p>
        <p>Best regards,</p>
        <p>DDF Department {$website_name}</p>
    </body>
    </html>";

    wp_mail($user_email, $subject, $message, $headers);
}
function trapz_hot_bank_send_debiting_email($user_id, $amount, $new_balance, $account_type) {
    $user_info = get_userdata($user_id);
    $user_email = $user_info->user_email;
    $user_display_name = $user_info->display_name;
    $currency_symbol = get_user_meta($user_id, 'currency', true);
    $formatted_amount = number_format((float)$amount, 2, '.', ',');
    $formatted_new_balance = number_format((float)$new_balance, 2, '.', ',');
    $website_name = get_bloginfo('name');
    $admin_email = get_bloginfo('admin_email');
    $account_type_name = ucfirst($account_type);

    $subject = "Direct Debit Notification - " . $website_name;
    $headers = array('Content-Type: text/html; charset=UTF-8');

    $message = "<!DOCTYPE html>
    <html lang='en'>
    <head>
        <meta charset='UTF-8'>
        <meta name='viewport' content='width=device-width, initial-scale=1.0'>
        <title>Direct Debit Notification</title>
    </head>
    <body>
        <p>Dear {$user_display_name},</p>
        <p>This is to notify you that a direct debit of <span style='color: red;'>{$currency_symbol}{$formatted_amount}</span> has been made from your {$account_type_name} account with {$website_name}. Please review the transaction details below:</p>
        <ul>
            <li><strong>Transaction Date:</strong> " . date('Y-m-d H:i:s') . "</li>
            <li><strong>Amount Debited:</strong> <span style='color: red;'>{$currency_symbol}{$formatted_amount}</span></li>
            <li><strong>{$account_type_name} Account Balance:</strong> <span style='color: green;'>{$currency_symbol}{$formatted_new_balance}</span></li>
        </ul>
        <p>If you have any questions or did not authorize this transaction, please contact our customer service team immediately at {$admin_email}.</p>
        <p>Thank you for your attention to this matter.</p>
        <p>Best regards,</p>
        <p>DDF Department {$website_name}</p>
    </body>
    </html>";

    wp_mail($user_email, $subject, $message, $headers);
}

function trapz_hot_bank_add_imf_code_field($user) {
    ?>
    <h3><?php _e("Extra Profile Information", "trapz-hot-bank"); ?></h3>

    <table class="form-table">
        <tr>
            <th><label for="imf_code"><?php _e("IMF Code"); ?></label></th>
            <td>
                <input type="text" name="imf_code" id="imf_code" value="<?php echo esc_attr(get_the_author_meta('imf_code', $user->ID)); ?>" class="regular-text" /><br />
                <span class="description"><?php _e("Please enter your IMF code."); ?></span>
            </td>
        </tr>
    </table>
    <?php
}
add_action('show_user_profile', 'trapz_hot_bank_add_imf_code_field'); // Adding to own profile
add_action('edit_user_profile', 'trapz_hot_bank_add_imf_code_field'); // Adding to other user profiles
function trapz_hot_bank_save_imf_code_field($user_id) {
    // Check for the capability to edit the user and the nonce for security
    if (!current_user_can('edit_user', $user_id) || !isset($_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'update-user_' . $user_id)) {
        return false;
    }

    // Update IMF code
    if (!empty($_POST['imf_code'])) {
        update_user_meta($user_id, 'imf_code', sanitize_text_field($_POST['imf_code']));
    }
}
add_action('personal_options_update', 'trapz_hot_bank_save_imf_code_field'); // Saving own profile
add_action('edit_user_profile_update', 'trapz_hot_bank_save_imf_code_field'); // Saving other user profiles

function trapz_hot_bank_transfer_form_shortcode() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to perform this action.';
    }

    ob_start();
    ?>
    <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
        <input type="hidden" name="action" value="trapz_hot_bank_handle_transfer">
        
        <!-- Transfer Form Fields -->
        <p><input type="text" name="recipient_name" placeholder="Recipient's Name" required /></p>
        <p><input type="text" name="recipient_account_number" placeholder="Recipient's Account Number" required /></p>
        <p><input type="text" name="recipient_bank_name" placeholder="Recipient's Bank Name" required /></p>
        <p><input type="text" name="recipient_address" placeholder="Recipient Address" required /></p>
        <p><input type="text" name="recipient_routing_number" placeholder="Recipient's Bank Routing Number" /></p>
        <p>
            <select name="your_account_info" required>
                <option value="savings">Savings Account</option>
                <option value="checking">Checking Account</option>
            </select>
        </p>
        <p><input type="number" name="amount_to_transfer" placeholder="Amount to Transfer" step="0.01" required /></p>
        <p><input type="text" name="currency" placeholder="Currency" required /></p>
        <p><input type="text" name="transfer_type" placeholder="Transfer Type (e.g., One-time, Recurring)" required /></p>
        <p><input type="text" name="purpose_of_transfer" placeholder="Purpose of Transfer" required /></p>
        <p><input type="date" name="date_of_transfer" placeholder="Date of Transfer" required /></p>
        <p><textarea name="reference_description" placeholder="Reference/Description"></textarea></p>
        <p><input type="submit" value="Transfer Funds" /></p>
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_transfer_form', 'trapz_hot_bank_transfer_form_shortcode');

function trapz_hot_bank_process_transfer_request() {
    if (!is_user_logged_in() || !isset($_POST['action']) || $_POST['action'] !== 'trapz_hot_bank_handle_transfer') {
        return;
    }

    global $wpdb;
    $pending_transfers_table = $wpdb->prefix . 'trapz_hot_bank_pending_transfers';
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    $recipient_name = sanitize_text_field($_POST['recipient_name']);
    $recipient_account_number = sanitize_text_field($_POST['recipient_account_number']);
    $amount_to_transfer = floatval($_POST['amount_to_transfer']);
    $account_type = sanitize_text_field($_POST['your_account_info']); // Ensure this field name matches your form
    $user_id = get_current_user_id();
    $currency = get_user_meta($user_id, 'currency', true);
    $imf_enabled = get_user_meta($user_id, 'trapz_hot_bank_imf_enabled', true);

    $account_balance = floatval(get_user_meta($user_id, "{$account_type}_account", true));
    if ($amount_to_transfer <= 0 || $amount_to_transfer > $account_balance) {
        wp_redirect(home_url('/insufficient-funds'));
        exit;
    }

    if ($imf_enabled === 'on') {
        // IMF validation is enabled for this user
        $wpdb->insert($pending_transfers_table, [
            'user_id' => $user_id,
            'recipient_name' => $recipient_name,
            'recipient_account_number' => $recipient_account_number,
            'account_type' => $account_type,
            'amount' => $amount_to_transfer,
            'currency' => $currency,
            'status' => 'pending'
        ]);

        wp_redirect(home_url('/imf-validation'));
    } else {
        // IMF validation is not enabled for this user
        // Directly process the transfer
        $new_balance = $account_balance - $amount_to_transfer;
        update_user_meta($user_id, "{$account_type}_account", $new_balance);

        $wpdb->insert($transactions_table, [
            'user_id' => $user_id,
            'reference' => 'Fund Transfer',
            'date' => current_time('mysql'),
            'type' => 'debit',
            'amount' => $amount_to_transfer,
            'currency' => $currency
        ]);

        // Optionally send a confirmation email
        trapz_hot_bank_send_transfer_confirmation_email($user_id, $amount_to_transfer, $account_type, $new_balance);
        wp_redirect(home_url('/thank-you'));
    }
    exit;
}
add_action('init', 'trapz_hot_bank_process_transfer_request');

function trapz_hot_bank_imf_validation_form_shortcode() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to perform this action.';
    }

    ob_start();
    ?>
<!DOCTYPE html>
<html>
<head>
    <title>IMF Validation Form</title>
    <style>
        /* Styles for the IMF Form Container */
        .imf-form-container {
            max-width: 400px;
            margin: 0 auto;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 4px;
            background-color: #f9f9f9;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        /* Styles for the Form Elements */
        .imf-form-container form {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }

        .imf-form-container input[type="text"],
        .imf-form-container input[type="submit"] {
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            font-size: 16px;
        }

        /* Style for the Submit Button */
        .imf-form-container input[type="submit"] {
            background-color: #16705A;
            color: white;
            cursor: pointer;
            transition: background-color 0.3s ease;
        }

        .imf-form-container input[type="submit"]:hover {
            background-color: #00A775;
        }
    </style>
</head>
<body>
    <div class="imf-form-container">
        <form method="post" action="your_form_processing_script.php">
            <input type="text" name="imf_code" placeholder="Enter Your IMF Code" required />
            <input type="hidden" name="action" value="trapz_hot_bank_validate_imf">
            <input type="submit" value="Validate Transfer" />
        </form>
    </div>
</body>
</html>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_imf_validation_form', 'trapz_hot_bank_imf_validation_form_shortcode');

function trapz_hot_bank_validate_imf_code() {
    if (!isset($_POST['action']) || $_POST['action'] !== 'trapz_hot_bank_validate_imf' || !is_user_logged_in()) {
        return;
    }

    global $wpdb;
    $user_id = get_current_user_id();
    $imf_code_input = sanitize_text_field($_POST['imf_code']);
    $user_imf_code = get_user_meta($user_id, 'imf_code', true);

    $pending_transfers_table = $wpdb->prefix . 'trapz_hot_bank_pending_transfers';
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    $pending_transfer = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$pending_transfers_table} 
         WHERE user_id = %d AND status = 'pending' 
         ORDER BY id DESC LIMIT 1", 
         $user_id
    ));

    if ($pending_transfer && $imf_code_input === $user_imf_code) {
        $account_balance = floatval(get_user_meta($user_id, "{$pending_transfer->account_type}_account", true));

        if ($account_balance >= $pending_transfer->amount) {
            $new_balance = $account_balance - $pending_transfer->amount;
            update_user_meta($user_id, "{$pending_transfer->account_type}_account", $new_balance);

            $wpdb->update($pending_transfers_table, ['status' => 'completed'], ['id' => $pending_transfer->id]);

            $wpdb->insert($transactions_table, [
                'user_id' => $user_id,
                'reference' => 'Fund Transfer',
                'date' => current_time('mysql'),
                'type' => 'debit',
                'amount' => $pending_transfer->amount,
                'currency' => $pending_transfer->currency
            ]);

            trapz_hot_bank_send_transfer_confirmation_email($user_id, $pending_transfer->amount, $pending_transfer->account_type, $new_balance);
            wp_redirect(home_url('/thank-you'));
        } else {
            wp_redirect(home_url('/insufficient-funds'));
        }
    } else {
        wp_redirect(home_url('/incorrect-imf-code'));
    }
    exit;
}
add_action('init', 'trapz_hot_bank_validate_imf_code');


function trapz_hot_bank_send_transfer_confirmation_email($user_id, $amount, $account_type, $new_balance) {
    $user_info = get_userdata($user_id);
    if (!$user_info) {
        return; // Exit if user data is not found
    }

    $user_email = $user_info->user_email;
    $display_name = $user_info->display_name;
    $currency_symbol = get_user_meta($user_id, 'currency', true); // Assuming you store the currency symbol in user meta
    $formatted_amount = number_format((float)$amount, 2, '.', ',');
    $formatted_new_balance = number_format((float)$new_balance, 2, '.', ',');
    $website_name = get_bloginfo('name');
    $admin_email = get_bloginfo('admin_email');

    $subject = "Transfer Confirmation - " . $website_name;
    $headers = array('Content-Type: text/html; charset=UTF-8');

    $message = "
    <html>
    <head>
        <title>Transfer Confirmation</title>
    </head>
    <body>
        <p>Dear {$display_name},</p>
        <p>We are pleased to inform you that your transfer has been successfully processed. The details are as follows:</p>
        <ul>
            <li>Amount Transferred: <span style='color: red;'>{$currency_symbol}{$formatted_amount}</span></li>
            <li>Remaining {$account_type} Account Balance: <span style='color: green;'>{$currency_symbol}{$formatted_new_balance}</span></li>
        </ul>
        <p>If you did not authorize this transaction, please contact us immediately at {$admin_email}.</p>
        <p>Thank you for using {$website_name}.</p>
        <p>Best Regards,</p>
        <p>{$website_name} Team</p>
    </body>
    </html>";

    wp_mail($user_email, $subject, $message, $headers);
}


function trapz_hot_bank_bill_pay_shortcode() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to perform this action.';
    }

    ob_start();
    ?>
    <div class="trapz-hot-bank-tabs">
        <ul class="tabs">
            <li><a href="#add-payee">Add Payee</a></li>
            <li><a href="#pay-bill">Pay Bill</a></li>
        </ul>

        <div id="add-payee">
            <!-- Form for Adding Payee -->
            <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
                <input type="hidden" name="action" value="trapz_hot_bank_add_payee">
                <p><input type="text" name="payee_name" placeholder="Payee Name" required /></p>
                <p><input type="text" name="payee_account_number" placeholder="Payee Account Number" required /></p>
                <p><input type="submit" value="Add Payee" /></p>
            </form>
        </div>

        <div id="pay-bill">
            <!-- Form for Paying Bill -->
            <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
                <input type="hidden" name="action" value="trapz_hot_bank_pay_bill">
                <p>
                    <select name="payee_select" required>
                        <option value="">Select Payee</option>
                        <?php trapz_hot_bank_get_payees_options(); ?>
                    </select>
                </p>
                <p><input type="number" name="amount" placeholder="Amount" step="0.01" required /></p>
                <p><input type="date" name="payment_date" required /></p>
                <p>
                    <select name="account_type" required>
                        <option value="savings">Savings Account</option>
                        <option value="checking">Checking Account</option>
                    </select>
                </p>
                <p><input type="submit" value="Pay Bill" /></p>
            </form>
        </div>
    </div>
    <style>
        .trapz-hot-bank-tabs .tabs { list-style-type: none; }
        .trapz-hot-bank-tabs .tabs li { display: inline; margin-right: 10px; }
        .trapz-hot-bank-tabs .tabs a { text-decoration: none; }
    </style>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_bill_pay', 'trapz_hot_bank_bill_pay_shortcode');
function trapz_hot_bank_get_payees_options() {
    $user_id = get_current_user_id();
    global $wpdb;
    $table_name = $wpdb->prefix . 'trapz_hot_bank_payees';
    $payees = $wpdb->get_results("SELECT * FROM $table_name WHERE user_id = $user_id");

    foreach ($payees as $payee) {
        echo "<option value='{$payee->id}'>{$payee->payee_name}</option>";
    }
}
function trapz_hot_bank_process_add_payee() {
    if (!isset($_POST['action']) || $_POST['action'] !== 'trapz_hot_bank_add_payee') {
        return;
    }

    // Sanitize input
    $payee_name = sanitize_text_field($_POST['payee_name']);
    $payee_account_number = sanitize_text_field($_POST['payee_account_number']);
    $user_id = get_current_user_id();

    // Insert into database
    global $wpdb;
    $table_name = $wpdb->prefix . 'trapz_hot_bank_payees';
    $wpdb->insert($table_name, [
        'user_id' => $user_id,
        'payee_name' => $payee_name,
        'payee_account_number' => $payee_account_number
    ]);
}
add_action('init', 'trapz_hot_bank_process_add_payee');
function trapz_hot_bank_process_pay_bill() {
    if (!isset($_POST['action']) || $_POST['action'] !== 'trapz_hot_bank_pay_bill') {
        return;
    }

    // Sanitize and validate input
    $payee_id = sanitize_text_field($_POST['payee_select']);
    $amount = floatval($_POST['amount']);
    $payment_date = sanitize_text_field($_POST['payment_date']);
    $account_type = sanitize_text_field($_POST['account_type']);
    $user_id = get_current_user_id();
    $currency = get_user_meta($user_id, 'currency', true);

    // Perform validation checks
    if ($amount <= 0) {
        // Handle invalid payment amount
        wp_redirect(home_url('/invalid-amount'));
        exit;
    }

    // Check balance and process payment
    $account_balance = floatval(get_user_meta($user_id, "{$account_type}_account", true));
    if ($account_balance >= $amount) {
        $new_balance = $account_balance - $amount;
        update_user_meta($user_id, "{$account_type}_account", $new_balance);

        // Record the transaction
        record_transaction($user_id, 'Bill Payment', 'debit', $amount, $currency);

        // Send email confirmation
        trapz_hot_bank_send_bill_payment_confirmation_email($user_id, $amount, $payee_id, $account_type);
        wp_redirect(home_url('/bill-paid'));
    } else {
        wp_redirect(home_url('/insufficient-funds'));
    }
    exit;
}

add_action('init', 'trapz_hot_bank_process_pay_bill');
function trapz_hot_bank_send_bill_payment_confirmation_email($user_id, $amount, $payee_id, $account_type) {
    $user_info = get_userdata($user_id);
    $user_email = $user_info->user_email;
    $currency_symbol = get_user_meta($user_id, 'currency', true);
    $formatted_amount = number_format($amount, 2, '.', ',');

    // Fetch payee name from database
    global $wpdb;
    $table_name = $wpdb->prefix . 'trapz_hot_bank_payees';
    $payee = $wpdb->get_row("SELECT * FROM $table_name WHERE id = $payee_id");

    $subject = "Bill Payment Confirmation";
    $headers = array('Content-Type: text/html; charset=UTF-8');
    $message = "
        <html>
<body>
    <p>Dear {$user_info->display_name},</p>
    <p>You have successfully paid <span style='color: red;'>{$currency_symbol}{$formatted_amount}</span> to {$payee->payee_name} from your {$account_type} account.</p>
    <p>Thank you for using our bill payment service.</p>
</body>
</html>";

    wp_mail($user_email, $subject, $message, $headers);
}


function trapz_hot_bank_add_payee_form_shortcode() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to perform this action.';
    }

    ob_start();
    ?>
    <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
        <input type="hidden" name="action" value="trapz_hot_bank_add_payee">
        <p><input type="text" name="payee_name" placeholder="Payee Name" required /></p>
        <p><input type="text" name="payee_account_number" placeholder="Payee Account Number" required /></p>
        <p><input type="submit" value="Add Payee" /></p>
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_add_payee_form', 'trapz_hot_bank_add_payee_form_shortcode');

function trapz_hot_bank_pay_bill_form_shortcode() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to perform this action.';
    }

    ob_start();
    ?>
    <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>">
        <input type="hidden" name="action" value="trapz_hot_bank_pay_bill">
        <p>
            <select name="payee_select" required>
                <option value="">Select Payee</option>
                <?php trapz_hot_bank_get_payees_options(); ?>
            </select>
        </p>
        <p><input type="number" name="amount" placeholder="Amount" step="0.01" required /></p>
        <p><input type="date" name="payment_date" required /></p>
        <p>
            <select name="account_type" required>
                <option value="savings">Savings Account</option>
                <option value="checking">Checking Account</option>
            </select>
        </p>
        <p><input type="submit" value="Pay Bill" /></p>
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_pay_bill_form', 'trapz_hot_bank_pay_bill_form_shortcode');



// Shortcode for displaying and processing the loan application form
add_shortcode('trapz_hot_bank_loan_application', 'trapz_hot_bank_display_and_process_loan_form');
function trapz_hot_bank_display_and_process_loan_form() {
    // Check if form is submitted and process it
    if (isset($_POST['action']) && $_POST['action'] === 'submit_loan_application') {
        trapz_hot_bank_handle_loan_submission();
    }

    // Display the form
    ob_start();
    ?>
    <!-- Loan Application Form -->
    <form id="loan-application-form" method="post" action="">
        <!-- Personal Information -->
        <h5>Personal Information</h5>
        <p><input type="text" name="full_name" placeholder="Full Name" required></p>
        <p><input type="date" name="date_of_birth" placeholder="Date of Birth" required></p>
        <p>
            <select name="identification_type" required>
                <option value="">Select Identification Type</option>
                <option value="social_security_number">Social Security Number</option>
                <option value="passport">Passport</option>
                <option value="driver_license">Driver's License</option>
            </select>
            <input type="text" name="identification_number" placeholder="Identification Number" required>
        </p>
        <p><input type="text" name="marital_status" placeholder="Marital Status" required></p>

        <!-- Contact Information -->
        <h5>Contact Information</h5>
        <p><input type="text" name="home_address" placeholder="Home Address" required></p>
        <p><input type="tel" name="phone_number" placeholder="Phone Number" required></p>
        <p><input type="email" name="email_address" placeholder="Email Address" required></p>

        <!-- Employment Information -->
        <h5>Employment Information</h5>
        <p>
            <select name="employment_status" required>
                <option value="">Select Employment Status</option>
                <option value="employed">Employed</option>
                <option value="self_employed">Self-Employed</option>
                <option value="unemployed">Unemployed</option>
            </select>
        </p>
        <p><input type="text" name="employer_name" placeholder="Employer Name"></p>
        <p><input type="text" name="job_title" placeholder="Job Title"></p>

        <!-- Financial Information -->
        <h5>Financial Information</h5>
        <p><input type="number" name="monthly_income" placeholder="Monthly Income" required></p>
        <p><input type="number" name="monthly_expenses" placeholder="Monthly Expenses" required></p>

        <!-- Loan Details -->
        <h5>Loan Details</h5>
        <p><input type="number" name="loan_amount" placeholder="Loan Amount Requested" required></p>
        <p>
            <select name="loan_purpose" required>
                <option value="">Select Purpose of the Loan</option>
                <option value="home_purchase">Home Purchase</option>
                <option value="car_purchase">Car Purchase</option>
                <option value="debt_consolidation">Debt Consolidation</option>
                <option value="education">Education</option>
                <option value="other">Other</option>
            </select>
        </p>
        <p>
            <select name="loan_term" required>
                <option value="">Select Loan Term</option>
                <option value="1">1 Year</option>
                <option value="2">2 Years</option>
                <option value="5">5 Years</option>
                <option value="10">10 Years</option>
                <option value="20">20 Years</option>
            </select>
        </p>

        <!-- Credit Information -->
        <h5>Credit Information</h5>
        <p>
            <select name="credit_score" required>
                <option value="">Select Credit Score</option>
                <option value="excellent">Excellent (720+)</option>
                <option value="good">Good (680-719)</option>
                <option value="fair">Fair (640-679)</option>
                <option value="poor">Poor (below 640)</option>
            </select>
        </p>

        <!-- Consent and Authorization -->
        <h5>Consent and Authorization</h5>
        <p><input type="checkbox" name="terms_and_conditions" required> I agree to the <a href="#">Terms and Conditions</a></p>
        <p><input type="checkbox" name="credit_authorization" required> I authorize the lender to check my credit history</p>

        <!-- Hidden Input for Action -->
        <input type="hidden" name="action" value="submit_loan_application">
        
        <!-- Submit Button -->
        <p><input type="submit" value="Submit Application"></p>
    </form>
    <?php
    return ob_get_clean();
}


function trapz_hot_bank_handle_loan_submission() {
    $data = $_POST; // Replace with proper sanitization and validation
    $current_user = wp_get_current_user();
    $data['user_email'] = $current_user->user_email;
    $data['display_name'] = $current_user->display_name;

    $approval_data = trapz_hot_bank_evaluate_loan_application($data);
    if ($approval_data['approved']) {
        update_user_meta($current_user->ID, 'trapz_hot_bank_last_approved_loan_amount', $approval_data['amount']);
        trapz_hot_bank_send_loan_approval_email($data, $approval_data['amount']);
        wp_redirect(home_url('/loan-approved'));
    } else {
        trapz_hot_bank_send_loan_decline_email($data);
        wp_redirect(home_url('/loan-declined'));
    }
    exit;
}

function trapz_hot_bank_evaluate_loan_application($data) {
    $approved = ($data['credit_score'] === 'excellent' || $data['credit_score'] === 'good');
    $approved_amount = rand(40000, 15000); // Random approved amount
    return ['approved' => $approved, 'amount' => $approved_amount];
}

function trapz_hot_bank_send_loan_approval_email($data, $approved_amount) {
    $currency_symbol = get_user_meta(get_current_user_id(), 'currency', true); 
    $website_name = get_bloginfo('name'); // Fetches the name of your WordPress website

    $subject = 'Loan Approval Notification';
    $message = "Dear " . $data['display_name'] . ",<br><br>" .
               "Your loan application has been approved for <span style='color: green;'>" . 
               $currency_symbol . number_format($approved_amount, 2) . "</span>.<br><br>" .
               "Your account will be credited within three business days.<br><br>" .
               "Thank you for using " . $website_name . " Loan Application Form.";
    $headers = ['Content-Type: text/html; charset=UTF-8'];

    wp_mail($data['user_email'], $subject, $message, $headers);

    // Notify admin
    $admin_email = get_option('admin_email');
    $admin_subject = 'User Loan Application Status';
    $admin_message = "User " . $data['display_name'] . " has been approved for a loan of " . $currency_symbol . number_format($approved_amount, 2) . ".";
    wp_mail($admin_email, $admin_subject, $admin_message, $headers);
}

function trapz_hot_bank_send_loan_decline_email($data) {
    $subject = 'Loan Application Status - Unfortunately, Not Approved';
    $message = "Dear " . $data['display_name'] . ",<br><br>We regret to inform you that your loan application has been declined.";
    $headers = ['Content-Type: text/html; charset=UTF-8'];
    wp_mail($data['user_email'], $subject, $message, $headers);

    // Notify admin
    $admin_email = get_option('admin_email');
    $admin_subject = 'User Loan Application Status';
    $admin_message = "User " . $data['display_name'] . "'s loan application has been declined.";
    wp_mail($admin_email, $admin_subject, $admin_message, $headers);
}

// Shortcode to display the approved loan amount
add_shortcode('trapz_hot_bank_approved_loan_amount', 'trapz_hot_bank_display_approved_loan_amount');
function trapz_hot_bank_display_approved_loan_amount() {
    $user_id = get_current_user_id();
    $approved_amount = get_user_meta($user_id, 'trapz_hot_bank_last_approved_loan_amount', true);
    $currency_symbol = get_user_meta($user_id, 'currency', true); 

    if (!$approved_amount) {
        return 'No approved loan amount available.';
    }

    return "<span style='color: green;'>{$currency_symbol}" . number_format($approved_amount, 2) . "</span>";
}

add_shortcode('trapz_hot_bank_mobile_deposit', 'trapz_hot_bank_mobile_deposit_form');
function trapz_hot_bank_mobile_deposit_form() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to use this feature.';
    }

    // Enqueue the style (Assuming you have the CSS in a file named 'style.css' in your plugin's directory)
    wp_enqueue_style('trapz-bank-css', plugin_dir_url(__FILE__) . 'style.css');

    ob_start();
    ?>
    <div class="trapz-bank-form">
        <form method="post" action="<?php echo esc_url($_SERVER['REQUEST_URI']); ?>" enctype="multipart/form-data">
            <label for="deposit_amount">Amount to Deposit:</label>
            <input type="number" name="deposit_amount" id="deposit_amount" placeholder="Enter amount" step="0.01" required />

            <label for="account_type">Account Type:</label>
            <select name="account_type" id="account_type" required>
                <option value="">Select Account Type</option>
                <option value="savings">Savings Account</option>
                <option value="checking">Checking Account</option>
                <option value="fixed_deposit">Fixed Deposit Account</option>
            </select>

            <label for="check_front">Upload Check Front:</label>
            <label class="custom-file-upload">
                <input type="file" name="check_front" id="check_front" required />
            </label>

            <label for="check_back">Upload Check Back:</label>
            <label class="custom-file-upload">
                <input type="file" name="check_back" id="check_back" required />
            </label>

            <input type="submit" value="Deposit Check" />
        </form>
    </div>
    <?php
    return ob_get_clean();
}


add_action('init', 'trapz_hot_bank_process_mobile_deposit');
function trapz_hot_bank_process_mobile_deposit() {
    if (isset($_FILES['check_front'], $_FILES['check_back']) && isset($_POST['deposit_amount'], $_POST['account_type'])) {
        // Process the uploaded files and validate the check (this part needs custom implementation)

        // Ensure the user is logged in
        if (!is_user_logged_in()) {
            // Handle the case where the user is not logged in
            return;
        }

        $user_id = get_current_user_id();
        $account_type = sanitize_text_field($_POST['account_type']);
        $deposit_amount = floatval($_POST['deposit_amount']);
        $currency = get_user_meta($user_id, 'currency', true);

        // Perform validation checks
        if ($deposit_amount <= 0) {
            // Handle invalid deposit amount
            return;
        }

        // Update user's account balance
        $current_balance = get_user_meta($user_id, "{$account_type}_account", true);
        $new_balance = $current_balance + $deposit_amount;
        update_user_meta($user_id, "{$account_type}_account", $new_balance);

        // Record the transaction
        record_transaction($user_id, 'Mobile Deposit', 'credit', $deposit_amount, $currency);

        // Send confirmation email
        trapz_hot_bank_send_deposit_email($user_id, $deposit_amount, $account_type, $new_balance);

        // Redirect to the deposit success page
        wp_redirect(home_url('/deposit-success'));
        exit;
    }
}


function trapz_hot_bank_send_deposit_email($user_id, $amount, $account_type, $new_balance) {
    $user_info = get_userdata($user_id);
    $user_email = $user_info->user_email;
    $currency_symbol = get_user_meta($user_id, 'currency', true);
    $formatted_amount = number_format($amount, 2, '.', ',');
    $formatted_new_balance = number_format($new_balance, 2, '.', ',');
    $website_name = get_bloginfo('name');
    
    $subject = "Check Deposit Confirmation - " . $website_name;
    $headers = array('Content-Type: text/html; charset=UTF-8');
    $message = "
        <html>
        <body>
            <p>Hello {$user_info->display_name},</p>
            <p>An amount of <span style='color: green;'>{$currency_symbol}{$formatted_amount}</span> has been credited to your {$account_type} account on " . date('Y-m-d H:i:s') . ". Your {$account_type} available balance is now <span style='color: green;'>{$currency_symbol}{$formatted_new_balance}</span>.</p>
            <p>Note: Even though your account is credited instantly, it's still under review. If the check is found to be inappropriate, the credited amount can be reversed, or it may lead to the closure of your account.</p>
            <p>Thanks for using {$website_name}.</p>
        </body>
        </html>";

    wp_mail($user_email, $subject, $message, $headers);
}

function record_transaction($user_id, $reference, $type, $amount, $currency) {
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    $data = array(
        'user_id' => $user_id,
        'reference' => $reference,
        'date' => current_time('mysql'),
        'type' => $type,
        'amount' => $amount,
        'currency' => $currency
    );
    $wpdb->insert($transactions_table, $data);
}

add_shortcode('trapz_hot_bank_account_history', 'trapz_hot_bank_display_account_history');
function trapz_hot_bank_display_account_history() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to view your account history.';
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';
    $transactions = $wpdb->get_results($wpdb->prepare("SELECT * FROM $transactions_table WHERE user_id = %d ORDER BY date DESC", $user_id));

    if(empty($transactions)) {
        return 'No transaction history available.';
    }

    ob_start();
    echo '<div class="trapz-bank-history">';
    echo '<table>';
    echo '<thead><tr><th>Reference</th><th>Date</th><th>Type</th><th>Amount</th></tr></thead>';
    echo '<tbody>';

    foreach ($transactions as $transaction) {
        $color = $transaction->type == 'credit' ? 'green' : 'red';
        echo '<tr>';
        echo '<td>' . esc_html($transaction->reference) . '</td>';
        echo '<td>' . esc_html(date('Y-m-d', strtotime($transaction->date))) . '</td>';
        echo '<td>' . esc_html(ucfirst($transaction->type)) . '</td>';
        echo '<td><span style="color: ' . $color . ';">' . esc_html($transaction->currency . number_format($transaction->amount, 2)) . '</span></td>';
        echo '</tr>';
    }

    echo '</tbody></table>';
    echo '</div>';
    return ob_get_clean();
}


function trapz_hot_bank_last_debit_amount() {
    if (!is_user_logged_in()) {
        return ''; // Returns an empty string if the user is not logged in.
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    // Query to get the last debit transaction
    $last_debit = $wpdb->get_row($wpdb->prepare("
        SELECT amount, currency 
        FROM $transactions_table 
        WHERE user_id = %d AND type = 'debit' 
        ORDER BY date DESC 
        LIMIT 1", 
        $user_id
    ));

    if ($last_debit) {
        // Returns only the currency symbol and the formatted amount.
        return esc_html($last_debit->currency) . number_format($last_debit->amount, 2);
    } else {
        return ''; // Returns an empty string if there are no debit transactions.
    }
}

add_shortcode('trapz_hot_bank_last_debit', 'trapz_hot_bank_last_debit_amount');

function trapz_hot_bank_last_credit_amount() {
    if (!is_user_logged_in()) {
        return ''; // Returns an empty string if the user is not logged in.
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    // Query to get the last credit transaction
    $last_credit = $wpdb->get_row($wpdb->prepare("
        SELECT amount, currency 
        FROM $transactions_table 
        WHERE user_id = %d AND type = 'credit' 
        ORDER BY date DESC 
        LIMIT 1", 
        $user_id
    ));

    if ($last_credit) {
        // Returns only the currency symbol and the formatted amount.
        return esc_html($last_credit->currency) . number_format($last_credit->amount, 2);
    } else {
        return ''; // Returns an empty string if there are no credit transactions.
    }
}

add_shortcode('trapz_hot_bank_last_credit', 'trapz_hot_bank_last_credit_amount');


function trapz_hot_bank_btc_purchase_form() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to purchase Bitcoin.';
    }

    $user_id = get_current_user_id();
    $message = '';

    if (isset($_POST['trapz_hot_bank_purchase_btc'])) {
        $btc_amount = floatval(sanitize_text_field($_POST['btc_amount']));
        $account_type = sanitize_text_field($_POST['account_type']);

        // Conversion rate: 1 unit of currency to BTC
        $conversion_rate = 0.000027; 
        $btc_to_purchase = $btc_amount * $conversion_rate;
        $current_balance = floatval(get_user_meta($user_id, "{$account_type}_account", true));

        if ($btc_amount <= $current_balance) {
            // Deduct the amount from the user's account
            $new_balance = $current_balance - $btc_amount;
            update_user_meta($user_id, "{$account_type}_account", $new_balance);

            // Record the BTC transaction
            trapz_hot_bank_record_btc_transaction($user_id, $btc_to_purchase);
            
            $message = 'Bitcoin purchase successful!';
        } else {
            $message = 'Insufficient funds in your account.';
        }
    }

    ob_start();
    ?>
    <div><?php echo esc_html($message); ?></div>
    <form method="post">
        <input type="number" name="btc_amount" min="500" placeholder="Enter amount" required>
        <select name="account_type">
            <option value="savings">Savings Account</option>
            <option value="checking">Checking Account</option>
        </select>
        <input type="submit" name="trapz_hot_bank_purchase_btc" value="Purchase Bitcoin">
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_btc_purchase', 'trapz_hot_bank_btc_purchase_form');

function trapz_hot_bank_record_btc_transaction($user_id, $btc_amount) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'trapz_hot_bank_btc_transactions';

    $wpdb->insert($table_name, array(
        'user_id' => $user_id,
        'btc_amount' => $btc_amount,
        'purchase_date' => current_time('mysql'),
        // Additional fields as required
    ));
}

function trapz_hot_bank_btc_transaction_history() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to view your BTC transaction history.';
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $table_name = $wpdb->prefix . 'trapz_hot_bank_btc_transactions';

    $transactions = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE user_id = %d ORDER BY purchase_date DESC", $user_id));

    ob_start();
    echo '<table>';
    // Table headers
    foreach ($transactions as $transaction) {
        // Display each transaction
    }
    echo '</table>';
    return ob_get_clean();
}
add_shortcode('trapz_hot_bank_btc_transaction_history', 'trapz_hot_bank_btc_transaction_history');


// Add the field to the user profile page
function trapz_hot_bank_add_imf_toggle_to_user_profile($user) {
    $imf_enabled = get_user_meta($user->ID, 'trapz_hot_bank_imf_enabled', true);
    ?>
    <h3><?php _e("IMF Feature Settings", "trapz-hot-bank"); ?></h3>

    <table class="form-table">
        <tr>
            <th><label for="trapz-hot-bank-imf-enabled"><?php _e("Enable IMF Validation"); ?></label></th>
            <td>
                <input type="checkbox" name="trapz_hot_bank_imf_enabled" id="trapz-hot-bank-imf-enabled" <?php checked($imf_enabled, 'on'); ?> />
                <span class="description"><?php _e("Check to enable IMF validation for this user."); ?></span>
            </td>
        </tr>
    </table>
    <?php
}
add_action('show_user_profile', 'trapz_hot_bank_add_imf_toggle_to_user_profile');
add_action('edit_user_profile', 'trapz_hot_bank_add_imf_toggle_to_user_profile');

// Save the field value
function trapz_hot_bank_save_imf_toggle($user_id) {
    if (!current_user_can('edit_user', $user_id)) {
        return false;
    }

    update_user_meta($user_id, 'trapz_hot_bank_imf_enabled', $_POST['trapz_hot_bank_imf_enabled']);
}
add_action('personal_options_update', 'trapz_hot_bank_save_imf_toggle');
add_action('edit_user_profile_update', 'trapz_hot_bank_save_imf_toggle');


function trapz_hot_bank_menu_item() {
    add_menu_page("Edit Transaction Dates", "Edit Transaction Dates", "manage_options", "trapz-hot-bank-edit-transaction-dates", "trapz_hot_bank_edit_transaction_dates_page", "dashicons-calendar-alt", 6);
}
add_action("admin_menu", "trapz_hot_bank_menu_item");

function trapz_hot_bank_edit_transaction_dates_page(){
    ?>
    <div class="wrap">
        <h1>Edit User Transaction Dates</h1>
        <form action="" method="post">
            <?php
            wp_nonce_field('trapz_hot_bank_edit_transactions_action', 'trapz_hot_bank_edit_transactions_nonce');
            wp_dropdown_users(array('name' => 'user_id', 'who' => 'subscriber', 'id' => 'trapz_hot_bank_user_dropdown')); 
            ?>
            <input type="submit" value="View Transactions" class="button-primary"/>
        </form>
        <?php trapz_hot_bank_display_user_transactions(); ?>
    </div>
    <?php
}
function trapz_hot_bank_display_user_transactions() {
    if (isset($_POST['user_id']) && check_admin_referer('trapz_hot_bank_edit_transactions_action', 'trapz_hot_bank_edit_transactions_nonce')) {
        global $wpdb;
        $user_id = intval($_POST['user_id']);
        $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';
        $transactions = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$transactions_table} WHERE user_id = %d", $user_id));

        if($transactions) {
            echo '<form method="post" action="">';
            wp_nonce_field('trapz_hot_bank_update_transactions_action', 'trapz_hot_bank_update_transactions_nonce');
            echo '<input type="hidden" name="user_id" value="' . esc_attr($user_id) . '"/>';
            foreach ($transactions as $transaction) {
                echo '<p><label for="date-' . esc_attr($transaction->id) . '">Transaction ID ' . esc_html($transaction->id) . ' (Date): </label>';
                echo '<input type="text" id="date-' . esc_attr($transaction->id) . '" name="transactions[' . esc_attr($transaction->id) . ']" value="' . esc_attr($transaction->date) . '"/></p>';
            }
            echo '<input type="submit" value="Update Dates" class="button-primary"/>';
            echo '</form>';
        }
    }
}
function trapz_hot_bank_update_transaction_dates() {
    if (isset($_POST['transactions'], $_POST['user_id']) && check_admin_referer('trapz_hot_bank_update_transactions_action', 'trapz_hot_bank_update_transactions_nonce')) {
        global $wpdb;
        $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';
        foreach ($_POST['transactions'] as $id => $date) {
            $wpdb->update($transactions_table, array('date' => sanitize_text_field($date)), array('id' => intval($id)));
        }
    }
}
add_action('admin_init', 'trapz_hot_bank_update_transaction_dates');


function trapz_hot_bank_cashflow_chart() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to view cash flow statistics.';
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    // Get total credits and debits
    $total_credits = $wpdb->get_var($wpdb->prepare("SELECT SUM(amount) FROM $transactions_table WHERE user_id = %d AND type = 'credit'", $user_id));
    $total_debits = $wpdb->get_var($wpdb->prepare("SELECT SUM(amount) FROM $transactions_table WHERE user_id = %d AND type = 'debit'", $user_id));

    // Prepare data for the chart
    $data = array(
        'credits' => floatval($total_credits),
        'debits' => floatval($total_debits)
    );

    // Output the container for the chart
    $output = '<canvas id="trapzHotBankCashFlowChart" width="400" height="400"></canvas>';
    $output .= '<script type="text/javascript">
        jQuery(document).ready(function($) {
            var ctx = document.getElementById("trapzHotBankCashFlowChart").getContext("2d");
            var cashFlowChart = new Chart(ctx, {
                type: "bar",
                data: {
                    labels: ["Credits", "Debits"],
                    datasets: [{
                        label: "Cash Flow",
                        data: [' . $data['credits'] . ', ' . $data['debits'] . '],
                        backgroundColor: ["#4CAF50", "#F44336"],
                        borderColor: ["#388E3C", "#D32F2F"],
                        borderWidth: 1
                    }]
                },
                options: {
                    scales: {
                        yAxes: [{
                            ticks: {
                                beginAtZero: true
                            }
                        }]
                    }
                }
            });
        });
    </script>';

    return $output;
}
add_shortcode('trapz_hot_bank_cashflow_chart', 'trapz_hot_bank_cashflow_chart');


function trapz_hot_bank_cashflow_line_chart() {
    if (!is_user_logged_in()) {
        return 'You must be logged in to view cash flow statistics.';
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $transactions_table = $wpdb->prefix . 'trapz_hot_bank_transactions';

    // Fetch transactions
    $transactions = $wpdb->get_results($wpdb->prepare("SELECT amount, type, date FROM $transactions_table WHERE user_id = %d ORDER BY date ASC", $user_id));

    // Prepare data for chart
    $data_points = array();
    foreach ($transactions as $transaction) {
        $point = array(
            'x' => $transaction->date,
            'y' => $transaction->type == 'credit' ? $transaction->amount : -$transaction->amount // Credits as positive, debits as negative
        );
        array_push($data_points, $point);
    }
    $data_points_json = json_encode($data_points);

    // Output the container for the chart
    $output = '<canvas id="trapzHotBankCashFlowLineChart" width="400" height="400"></canvas>';
    $output .= '<script type="text/javascript">
        jQuery(document).ready(function($) {
            var ctx = document.getElementById("trapzHotBankCashFlowLineChart").getContext("2d");
            var cashFlowChart = new Chart(ctx, {
                type: "line",
                data: {
                    datasets: [{
                        label: "Cash Flow",
                        data: ' . $data_points_json . ',
                        backgroundColor: "rgba(75, 192, 192, 0.2)",
                        borderColor: "rgba(75, 192, 192, 1)",
                        borderWidth: 1
                    }]
                },
                options: {
                    scales: {
                        xAxes: [{
                            type: "time",
                            time: {
                                unit: "day"
                            }
                        }],
                        yAxes: [{
                            ticks: {
                                beginAtZero: true
                            }
                        }]
                    }
                }
            });
        });
    </script>';

    return $output;
}
add_shortcode('trapz_hot_bank_cashflow_line_chart', 'trapz_hot_bank_cashflow_line_chart');